//
//  ItemProviders.swift
//  Do It
//
//  Created by Jim Dovey on 3/22/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import Foundation
import CoreServices

// START:UTIs
let todoItemUTI = "com.pragprog.swiftui.todo.item" //<label id="code.8.item.uti"/>
let todoListUTI = "com.pragprog.swiftui.todo.list" //<label id="code.8.list.uti"/>
let todoItemUUIDUTI = "com.pragprog.swiftui.todo.item.uuid" //<label id="code.8.item.uuid.uti"/>
let todoListUUIDUTI = "com.pragprog.swiftui.todo.list.uuid" //<label id="code.8.list.uuid.uti"/>
let rawTextUTI = kUTTypeUTF8PlainText as String //<label id="code.8.text.uti"/>
let jsonUTI = kUTTypeJSON as String //<label id="code.8.json.uti"/>
// END:UTIs

/// Special variants of `TodoItem` and `TodoItemList` which use a hierarchical
/// format for export and import as JSON.
fileprivate enum Serialized {
    struct Item: Codable {
        var id: UUID
        var title: String
        var priority: TodoItem.Priority
        var notes: String?
        var date: Date?
        var completed: Date?
        
        init(_ item: TodoItem) {
            self.id = item.id
            self.title = item.title
            self.priority = item.priority
            self.notes = item.notes
            self.date = item.date
            self.completed = item.completed
        }
    }
    
    struct List: Codable {
        var id: UUID
        var name: String
        var color: TodoItemList.Color
        var icon: String
        var items: [Item]
        
        init(_ list: TodoItemList, _ items: [TodoItem]) {
            self.id = list.id
            self.name = list.name
            self.color = list.color
            self.icon = list.icon
            self.items = items.map(Item.init)
        }
    }
}

/// Errors thrown by the item provider/receiver implementation.
enum ItemProviderError: CustomNSError {
    /// Attempting to import an item by UUID, but UUID was not in the data store.
    case itemNotFound(UUID)
    
    /// Attempting to import a list by UUID, but UUID was not in the data store.
    case listNotFound(UUID)
    
    /// Failed to encode a string using UTF-8. This is theoretically impossible, but
    /// `String.data(using: .utf8)` returns an `Optional`, so we have
    /// to handle `nil` results somehow.
    ///
    /// Contains the string that couldn't be encoded.
    case stringEncodingFailure(String)
    
    /// `NSItemProvider` asked us to create or read data using a UTI that we
    /// don't recognize. Contains the offending UTI along with the UTIs we *do*
    /// support.
    case unsupportedUTI(String, [String])
    
    /// A provider didn't use any UTIs we understand.
    case noUTIsSupported(provided: [String], understood: [String])
    
    /// The string for an item/list UUID was not a valid UUID.
    case invalidUUIDString(String)
    
    /// We weren't able to decode the given data using any of the usual string
    /// encodings. We try UTF-8 for UUID strings, and UTF-8, UTF-16, and
    /// UTF-16BE for raw text import.
    case unknownStringEncoding(Data)
    
    /// No data was provided while receiving items.
    case noData
    
    /// A composite of several different errors.
    case multipleErrors([Error])
    
    /// The vended type of data was not appropriate for the drop target.
    case unsupportedDataType
}

fileprivate let itemUnavailableError: NSError =
    NSError(domain: NSItemProvider.errorDomain,
            code: NSItemProvider.ErrorCode.itemUnavailableError.rawValue,
            userInfo: nil)

// MARK: - NSUserActivity

extension TodoItem {
    static let activityType = "com.pragprog.swiftui.ShowTodoItem"
    
    var userActivity: NSUserActivity {
        let activity = NSUserActivity(activityType: Self.activityType)
        activity.title = title
        activity.targetContentIdentifier = id.uuidString
        return activity
    }
}

extension TodoItemList {
    static let activityType = "com.pragprog.swiftui.ShowTodoList"
    
    // START:UserActivity
    var userActivity: NSUserActivity {
        let activity = NSUserActivity(activityType: Self.activityType)
        activity.title = name
        activity.targetContentIdentifier = id.uuidString
        return activity
    }
    // END:UserActivity
}

// MARK: - Sending Data

final class TodoItemProvider: NSObject, NSItemProviderWriting {
    let dataCenter: DataCenter
    let content: Either<TodoItem, TodoItemList>
    
    /// Creates an `ItemProvider` to encode a single to-do item.
    /// - Parameters:
    ///   - dataCenter: The shared `DataCenter` with all items and lists.
    ///   - item: The item to provide.
    init(dataCenter: DataCenter, item: TodoItem) {
        self.dataCenter = dataCenter
        self.content = .left(item)
    }
    
    /// Creates an `ItemProvider` to encode a full to-do list.
    /// - Parameters:
    ///   - dataCenter: The shared `DataCenter` with all items and lists.
    ///   - list: The list to provide.
    init(dataCenter: DataCenter, list: TodoItemList) {
        self.dataCenter = dataCenter
        self.content = .right(list)
    }
    
    public class var writableTypeIdentifiersForItemProvider: [String] {
        [rawTextUTI]
    }
    
    public var writableTypeIdentifiersForItemProvider: [String] {
        switch content {
        case .left:
            return [todoItemUUIDUTI, todoItemUTI, rawTextUTI]
        case .right:
            return [todoListUUIDUTI, todoListUTI, rawTextUTI]
        }
    }
    
    public static func itemProviderVisibilityForRepresentation(
        withTypeIdentifier typeIdentifier: String
    ) -> NSItemProviderRepresentationVisibility {
        switch typeIdentifier {
        case todoItemUUIDUTI, todoListUUIDUTI:
            // Must be looking at the same data store to handle this.
            return .ownProcess
        case todoItemUTI, todoListUTI:
            // Our apps know how to parse this, so `.team` would be
            // appropriate.
            // To allow generic JSON export, change to .all and see
            // line 215. Note that Messages likes JSON, but not ours.
            #if os(macOS)
            return .group
            #else
            return .team
            #endif
        default:
            // The only option left is UTF-8 plain text, which is
            // specifically for other apps.
            return .all
        }
    }
    
    func loadData(
        withTypeIdentifier typeIdentifier: String,
        forItemProviderCompletionHandler completionHandler: @escaping (Data?, Error?) -> Void
    ) -> Progress? {
        let progress = Progress(totalUnitCount: 1)
        
        // The NSItemProvider API is designed for background processing.
        DispatchQueue.global(qos: .utility).async {
            do {
                let data: Data
                
                // See if the requested UTI matches up to the type of data we
                // actually have.
                switch (self.content, typeIdentifier) {
                case (_, rawTextUTI):
                    // supported by both types
                    data = try self.plainTextRepresentation()
                    
                case (.left(let item), todoItemUUIDUTI):
                    guard let uuidData = item.id.uuidString.data(using: .utf8) else {
                        throw ItemProviderError.stringEncodingFailure(item.id.uuidString)
                    }
                    data = uuidData
                case (.right(let list), todoListUUIDUTI):
                    guard let uuidData = list.id.uuidString.data(using: .utf8) else {
                        throw ItemProviderError.stringEncodingFailure(list.id.uuidString)
                    }
                    data = uuidData
                case (.left, todoItemUTI), (.right, todoListUTI): // (_, jsonUTI):
                    // If a particular item/list format is requested, ensure we
                    // have that type of data.
                    data = try self.jsonRepresentation()
                    
                default:
                    throw ItemProviderError.unsupportedUTI(typeIdentifier, self.writableTypeIdentifiersForItemProvider)
                }
                
                progress.completedUnitCount = 1
                completionHandler(data, nil)
            }
            catch {
                progress.completedUnitCount = 1
                completionHandler(nil, error)
            }
        }
        
        return progress
    }
    
    /// Produces JSON data in UTF-8 encoding.
    private func jsonRepresentation() throws -> Data {
        switch content {
        case .left(let item):
            return try JSONEncoder().encode(Serialized.Item(item))
        case .right(let list):
            return try JSONEncoder().encode(Serialized.List(list, dataCenter.items(in: list)))
        }
    }
    
    /// Produces a plain-text representation of the item or list.
    private func plainTextRepresentation() throws -> Data {
        switch content {
        case .left(let item):
            return try plainTextItemRepresentation(item)
        case .right(let list):
            return try plainTextListRepresentation(list)
        }
    }
    
    /// Plain-text representation of a to-do item. Optionally includes due date, completion
    /// date, and any attached notes.
    ///
    /// Example:
    ///
    ///     """
    ///     My To-do Item
    ///     Default List
    ///
    ///     Priority: normal
    ///     Completed: 3/10/20 12:43 PM
    ///     Due Date: 3/12/20 10:00 PM
    ///
    ///     Item Notes...
    ///     """
    ///
    /// - Parameter item: The item being represented.
    /// - Throws: `ItemProviderError.stringEncodingFailure` if the string
    /// could not be formatted as UTF-8.
    /// - Returns: UTF-8 encoded plain-text.
    private func plainTextItemRepresentation(_ item: TodoItem) throws -> Data {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        formatter.timeStyle = .short
        
        let list = dataCenter.list(for: item)
        
        var str = """
            \(item.title)
            \(list.name)
            
            Priority: \(item.priority.rawValue)
            """
        
        if let completed = item.completed {
            str += "\nCompleted: \(formatter.string(from: completed))"
        }
        if let date = item.date {
            str += "\nDue Date: \(formatter.string(from: date))"
        }
        if let notes = item.notes {
            str += "\n\n\(notes)"
        }
        
        guard let data = str.data(using: .utf8) else {
            throw ItemProviderError.stringEncodingFailure(str)
        }
        return data
    }
    
    /// Plain-text representation of a to-do list. Contains the list name and each item's name
    /// accompanied by a checkbox indicating the item's completion status.
    ///
    /// Example:
    ///
    ///     """
    ///     My List
    ///     ☐ Incomplete Item
    ///     ☑︎ Complete Item
    ///     """
    ///
    /// - Parameter list: The to-do list being represented.
    /// - Throws: `ItemProviderError.stringEncodingFailure` if the string
    /// could not be formatted as UTF-8.
    /// - Returns: UTF-8 encoded plain-text.
    private func plainTextListRepresentation(_ list: TodoItemList) throws -> Data {
        let items = dataCenter.items(in: list)
            .sorted { $0.title.localizedStandardCompare($1.title) == .orderedAscending }
        let itemStrings = items.lazy.map {
            " \($0.complete ? "☑︎" : "☐") \($0.title)"
        }
        let str = "\(list.name)\n\(itemStrings.joined(separator: "\n"))"
        
        guard let data = str.data(using: .utf8) else {
            throw ItemProviderError.stringEncodingFailure(str)
        }
        return data
    }
}

// MARK: - Receiving Data

/// A class used to parse data from an array of `NSItemProvider` instances.
final class ItemReceiver {
    enum Output {
        /// A single new to-do item.
        ///
        /// Created from `com.pragprog.swiftui.todo.item` data.
        case item(TodoItem)
        /// A new to-do list with zero or more items.
        ///
        /// Created from `com.pragprog.swiftui.todo.list` data.
        case list(TodoItemList, [TodoItem])
        /// An item from the same data store was dropped; this is a move operation.
        ///
        /// Created from `com.pragprog.swiftui.todo.item.uuid` data.
        case existingItem(TodoItem)
        /// A list from the same data store was dropped; this is a move operation.
        ///
        /// Created from `com.pragprog.swiftui.todo.list.uuid` data.
        case existingList(TodoItemList)
        /// A string value, which can be used as the title of a new item or list.
        ///
        /// Created from `public.utf8-plain-text` data.
        case string(String)
    }
    
    private var dataCenter: DataCenter
    
    private static var understoodUTIs = [
        todoItemUUIDUTI, todoListUUIDUTI,
        todoItemUTI, todoListUTI,
        rawTextUTI, jsonUTI
    ]
    
    init(dataCenter: DataCenter) {
        self.dataCenter = dataCenter
    }
    
    /// Asynchronously decodes the data from the given list of providers, sending
    /// the resulting `Output` items and any error to a completion.
    ///
    /// - note: The `completion` block will be invoked on the main thread.
    ///
    /// - Parameters:
    ///   - providers: A list of item providers from a drop operation.
    ///   - completion: A completion block that will receive the output and/or error.
    /// - Returns: A `Progress` object that can be used to monitor the decode process.
    @discardableResult
    func readFromProviders(_ providers: [NSItemProvider], completion: @escaping ([Output], Error?) -> Void) -> Progress? {
        let group = DispatchGroup()
        var decoded: [Output?] = Array(repeatElement(nil, count: providers.count))
        var errors: [Error] = []
        
        let progress = Progress(totalUnitCount: Int64(providers.count) * 2)
        
        func handleResult(for index: Int, progress: Progress, block: () throws -> Output) {
            switch Result(catching: block) {
            case .success(let output):
                decoded[index] = output
            case .failure(let error):
                errors.append(error)
            }
            
            progress.completedUnitCount = progress.totalUnitCount
            group.leave()
        }
        
        for (index, provider) in providers.enumerated() {
            group.enter()
            let itemProgress = Progress(totalUnitCount: 1, parent: progress, pendingUnitCount: 1)
            
            // Select a UTI to use, in order of preference.
            if provider.hasItemConformingToTypeIdentifier(todoItemUUIDUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: todoItemUUIDUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleItemUUID(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else if provider.hasItemConformingToTypeIdentifier(todoListUUIDUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: todoListUUIDUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleListUUID(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else if provider.hasItemConformingToTypeIdentifier(todoItemUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: todoItemUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleItemJSON(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else if provider.hasItemConformingToTypeIdentifier(todoListUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: todoItemUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleListJSON(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else if provider.hasItemConformingToTypeIdentifier(rawTextUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: todoItemUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleRawText(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else if provider.hasItemConformingToTypeIdentifier(jsonUTI) {
                let loadProgress = provider.loadDataRepresentation(forTypeIdentifier: jsonUTI) { data, error in
                    handleResult(for: index, progress: itemProgress) {
                        try self.handleUnknownJSON(data: data, error: error)
                    }
                }
                progress.addChild(loadProgress, withPendingUnitCount: 1)
            }
            else {
                errors.append(ItemProviderError.noUTIsSupported(
                    provided: provider.registeredTypeIdentifiers,
                    understood: Self.understoodUTIs))
                group.leave()
            }
        }
        
        group.notify(queue: .main) {
            let error: Error?
            if errors.isEmpty {
                error = nil
            }
            else if errors.count == 1 {
                error = errors.first!
            }
            else {
                error = ItemProviderError.multipleErrors(errors)
            }
            
            completion(decoded.compactMap { $0 }, error)
        }
        
        return progress
    }
    
    private func handleItemUUID(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        
        guard let str = String(data: data, encoding: .utf8) else {
            throw ItemProviderError.unknownStringEncoding(data)
        }
        guard let uuid = UUID(uuidString: str) else {
            throw ItemProviderError.invalidUUIDString(todoItemUUIDUTI)
        }
        guard let item = dataCenter.item(withID: uuid) else {
            throw ItemProviderError.itemNotFound(uuid)
        }
        
        return .existingItem(item)
    }
    
    private func handleListUUID(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        
        guard let str = String(data: data, encoding: .utf8) else {
            throw ItemProviderError.unknownStringEncoding(data)
        }
        guard let uuid = UUID(uuidString: str) else {
            throw ItemProviderError.invalidUUIDString(todoListUUIDUTI)
        }
        guard let list = dataCenter.list(withID: uuid) else {
            throw ItemProviderError.listNotFound(uuid)
        }
        
        return .existingList(list)
    }
    
    private func handleItemJSON(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        
        let decoder = JSONDecoder()
        let item = try decoder.decode(Serialized.Item.self, from: data)
        let newItem = TodoItem(title: item.title,
                               priority: item.priority,
                               notes: item.notes,
                               date: item.date,
                               listID: UUID(),
                               completed: item.completed)
        
        return .item(newItem)
    }
    
    private func handleListJSON(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        
        let decoder = JSONDecoder()
        let list = try decoder.decode(Serialized.List.self, from: data)
        let newList = TodoItemList(name: list.name,
                                   color: list.color,
                                   icon: list.icon)
        let items = list.items.map {
            TodoItem(title: $0.title,
                     priority: $0.priority,
                     notes: $0.notes,
                     date: $0.date,
                     listID: newList.id,
                     completed: $0.completed)
        }
        
        return .list(newList, items)
    }
    
    private func handleUnknownJSON(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        
        if let listOutput = try? self.handleListJSON(data: data, error: error) {
            return listOutput
        }
        else if let itemOutput = try? self.handleItemJSON(data: data, error: error) {
            return itemOutput
        }
        else {
            throw ItemProviderError.unsupportedDataType
        }
    }
    
    private func handleRawText(data: Data?, error: Error?) throws -> Output {
        guard let data = data, !data.isEmpty else {
            throw error ?? ItemProviderError.noData
        }
        guard let str = String(data: data, encoding: .utf8)
            ?? String(data: data, encoding: .utf16)
            ?? String(data: data, encoding: .utf16BigEndian) else {
                throw ItemProviderError.unknownStringEncoding(data)
        }
        
        return .string(str)
    }
}
